/** @file   modifyevent.h
 * @brief   Declaration of ModifyEvent - classes.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_MODIFYEVENT_H
#define H_WWW_MODIFYEVENT_H

#include "dynamicevent.h"


namespace WeWantWar {


/** @class  ModifyEvent
 * @brief   Modifies the map-contents during the gameplay.
 * @author  Tomi Lamminsaari
 *
 * This event modifies the Map contents. You can set new tiles to the map or
 * copy an area of blocks from place to another and from map layer to another.
 */
class ModifyEvent : public DynamicEvent
{
public:

  ///
  /// Static members and structures.
  /// ==============================
  
  /** This structure holds the information about how the blocks in the
   * map should be changes.
   */
  struct ModifyItem {
    /** X-position in blocks */
    int blockX;
    /** Y-position in blocks. */
    int blockY;
    /** Target layer */
    int layer;
    /** New block-data. */
    int newBlock;
  };
  

  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Constructor.
   */
  ModifyEvent();
  
  /** Destructor.
   */
  virtual ~ModifyEvent();
  
  
  
  ///
  /// Methods
  /// =======
  
  /** Reimplemented readData-method.
   * @param     fin               The input stream
   * @return    0 if successful. Nonzero otherwise.
   */
  virtual int readData( std::istream& fin );
  
  /** Activates this event
   */
  virtual void activate();
  
  /** Reimplemented update-method.
   */
  virtual void update();
  
  
protected:

  ///
  /// Protected methods
  /// =================
  
  /** Creates and loads a single modification.
   * @param     fin               Input stream
   */
  void createModification( std::istream& fin );
  
  /** Creates a horizontal line modification.
   * @param     fin               Input stream
   */
  void createHLineModification( std::istream& fin );
  
  /** Creates a vertical line modification.
   * @param     fin               Input stream
   */
  void createVLineModification( std::istream& fin );
  
  /** Reads and creates an modification that copies the map data from
   * somewhere to another place.
   * @param     rIn               Input stream
   * @return    Nonzero if fails
   */
  int readCopyBlock( std::istream& rIn );
  
  /** Reads and creates the modification that fills a certain area of the
   * tilemap by using a certain tile.
   * @param     rIn               Input stream
   * @return    Nonzero if fails.
   */
  int readAreaFill( std::istream& rIn );
  
  /** Does the modifications to the map.
   */
  void doModifications();
  
  
  ///
  /// Members
  /// =======
  
  /** This vector holds all the modifications this event should make. */
  std::vector<ModifyItem*>  m_modifications;
};

};  // end of namespace

#endif // MODIFYEVENT_H

/**
 * Version history
 * ===============
 * $Log: modifyevent.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:06+02  lamminsa
 * Initial revision
 *
 */
 
